/* Call Library source file */
#include "LabJackLabviewLibrary.h"
#include "labjackusb.h"
#include "u6.h"

#define READ_REGISTER_SEND_LENGTH 14
#define READ_DOUBLE_REGISTER_REC_LENGTH 13
#define READ_SINGLE_REGISTER_REC_LENGTH 11

#define WRITE_DOUBLE_REGISTER_SEND_LENGTH 19
#define WRITE_SINGLE_REGISTER_SEND_LENGTH 14
#define WRITE_REGISTER_REC_LENGTH 12

// Function Prototypes


// Begin Function definitions

int openDevice(int DeviceNumber, int ProductType, int *error)
{
	*error = 0; // No error
	HANDLE devHandle = 0;
	
	devHandle = LJUSB_OpenDevice(DeviceNumber, 0, ProductType);
	
	if( devHandle == 0 ) {
		*error = -1;
		return 0;
	}
	
	return (int)devHandle;
}

void closeDevice(int devHandle){
	LJUSB_CloseDevice((HANDLE)devHandle);
}

int lj_read(int devHandle, int aSize, unsigned char* recArray){
	int i, r;
	
	BYTE recBuffer[aSize];
	
	r = LJUSB_Read( (HANDLE)devHandle, recBuffer, aSize );
	
	if (r == aSize) {
		for (i = 0; i < aSize; i++) {
			recArray[i] = recBuffer[i];
		}
	} else {
		return -1;
	}
	
	
	return 0;
}

int lj_write(int devHandle, int aSize, unsigned char* sendArray){
	int i, r;
	BYTE writeBuffer[aSize];
	
	for (i = 0; i < aSize; i++) {
		writeBuffer[i] = (BYTE)sendArray[i];
	}
	
	r = LJUSB_Write( (HANDLE)devHandle, writeBuffer, aSize );
	
	if( r == aSize){
		return 0;
	} else {
		return -1;
	}
}

int eAIN_U6(int deviceHandle, int channelP, int channelN, double* voltage, int range, int resolution, int settling, int binary) {
    int r = 0; // For checking return values
    u6CalibrationInfo caliInfo; // For calibration info
    HANDLE devHandle = (HANDLE)deviceHandle;
    
    r = getCalibrationInfo(devHandle, &caliInfo);
    
    if(r < 0) {
        return r;
    }
    
    double dblVoltage;
    r = eAIN(devHandle, &caliInfo, channelP, channelN, &dblVoltage, range, resolution, settling, binary, 0, 0 );
    
    if( r != 0 ){
        return r;
    }
    
    *voltage = dblVoltage;
    
	
	return 0;
}

int eTCConfig_U6(int deviceHandle, long* aEnableTimers, long* aEnableCounters, int TCPinOffset, int TimerClockBaseIndex, int TimerClockDivisor, long* aTimerModes, double* aTimerValues){
    int r = 0; // For checking return values
    HANDLE devHandle = (HANDLE)deviceHandle;
    
    r = eTCConfig(devHandle, aEnableTimers, aEnableCounters, TCPinOffset, TimerClockBaseIndex, TimerClockDivisor, aTimerModes, aTimerValues, 0, 0);
	
	return r;
}

int eTCValues_U6(int deviceHandle, long* aReadTimers, long* aUpdateResetTimers, long* aReadCounters, long* aResetCounters, double* aTimerValues, double* aCounterValues) {
    int r = 0; // For checking return values
    HANDLE devHandle = (HANDLE)deviceHandle;
    
    r = eTCValues(devHandle, aReadTimers, aUpdateResetTimers, aReadCounters, aResetCounters, aTimerValues, aCounterValues, 0, 0);
	
	return r;
}


int writeFloatingPointRegister(int deviceHandle, int Register, float *Value){
	int r;
	
    if(deviceHandle == 0) {
        return -1; // No device handle, just pass.
    }
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[WRITE_DOUBLE_REGISTER_SEND_LENGTH], recBuffer[WRITE_REGISTER_REC_LENGTH];
    
    float fValue = (float)*Value;
    BYTE* fPointer; // pointer for casting float to bytes.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 56;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 11;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Write Registers ( function # 16 )
	sendBuffer[9] = 16;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	// Bytes 10 and 11 are NumRegs, in our case always 2
	sendBuffer[12] = 0;
	sendBuffer[13] = 2;
	
	// Byte 12 is Byte Count, in our case, always 4
	sendBuffer[14] = 4;
	
	// Bytes 13-16 are the floating point value
	fPointer = (BYTE*)&fValue;
	
	sendBuffer[15] = fPointer[3];
	sendBuffer[16] = fPointer[2];
	sendBuffer[17] = fPointer[1];
	sendBuffer[18] = fPointer[0];
	
	
	r = LJUSB_Write( devHandle, sendBuffer, WRITE_DOUBLE_REGISTER_SEND_LENGTH );
    
    if( r != WRITE_DOUBLE_REGISTER_SEND_LENGTH ) {
        return -2; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, WRITE_REGISTER_REC_LENGTH );
	
    if( r != WRITE_REGISTER_REC_LENGTH ) {
        return -3; //Error in read
    }
	
	return 0;
}


int writeIntRegister(int deviceHandle, int Register, int *Value) {
	int r;
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[WRITE_DOUBLE_REGISTER_SEND_LENGTH], recBuffer[WRITE_REGISTER_REC_LENGTH];
    
    int iValue = *Value;
    BYTE* iPointer; // pointer for casting float to bytes.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 58;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 11;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Write Registers ( function # 16 )
	sendBuffer[9] = 16;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	// Bytes 10 and 11 are NumRegs, in int's case: 2
	sendBuffer[12] = 0;
	sendBuffer[13] = 2;
	
	// Byte 12 is Byte Count, in our case, always 4
	sendBuffer[14] = 4;
	
	// Bytes 13-16 are the bytes of the int value
	iPointer = (BYTE*)&iValue;
	
	sendBuffer[15] = iPointer[3];
	sendBuffer[16] = iPointer[2];
	sendBuffer[17] = iPointer[1];
	sendBuffer[18] = iPointer[0];
	
	
	r = LJUSB_Write( devHandle, sendBuffer, WRITE_DOUBLE_REGISTER_SEND_LENGTH );
    
    if( r != WRITE_DOUBLE_REGISTER_SEND_LENGTH ) {
        return -1; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, WRITE_REGISTER_REC_LENGTH );
	
    if( r != WRITE_REGISTER_REC_LENGTH ) {
        return -2; //Error in read
    }
	
	return 0;	
}

int writeShortRegister(int deviceHandle, int Register, short *Value) {
	int r;
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[WRITE_SINGLE_REGISTER_SEND_LENGTH], recBuffer[WRITE_REGISTER_REC_LENGTH];
    
    short sValue = (short)*Value;
    BYTE* sPointer; // pointer for casting a short to bytes.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 60;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 6;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Write Register ( function # 6 )
	sendBuffer[9] = 6;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	// Bytes 10-11 are the short value
	sPointer = (BYTE*)&sValue;
	
	sendBuffer[12] = sPointer[1];
	sendBuffer[13] = sPointer[0];
	
	
	r = LJUSB_Write( devHandle, sendBuffer, WRITE_SINGLE_REGISTER_SEND_LENGTH );
    
    if( r != WRITE_SINGLE_REGISTER_SEND_LENGTH ) {
        return -1; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, WRITE_REGISTER_REC_LENGTH );
	
    if( r != WRITE_REGISTER_REC_LENGTH ) {
        return -2; //Error in read
    }
	
	return 0;	
}

int readFloatingPointRegister(int deviceHandle, int Register, float* Value) {
	int r;
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[READ_REGISTER_SEND_LENGTH], recBuffer[READ_DOUBLE_REGISTER_REC_LENGTH], fBuffer[4];
    
    float* fPointer; // pointer for casting bytes to float.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 55;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 6;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Read Holding Registers ( function # 3 )
	sendBuffer[9] = 3;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	//Bytes 10 and 11 are NumRegs, in our case always 2
	sendBuffer[12] = 0;
	sendBuffer[13] = 2;
	
	r = LJUSB_Write( devHandle, sendBuffer, READ_REGISTER_SEND_LENGTH );
    
    if( r != READ_REGISTER_SEND_LENGTH ) {
        return -1; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, READ_DOUBLE_REGISTER_REC_LENGTH );
	
    if( r != READ_DOUBLE_REGISTER_REC_LENGTH ) {
        return -2; //Error in read
    }
    
    fBuffer[0] = recBuffer[12];
    fBuffer[1] = recBuffer[11];
    fBuffer[2] = recBuffer[10];
    fBuffer[3] = recBuffer[9];
    
    fPointer = (float*)fBuffer;
	
    *Value = *fPointer;
	
	return 0;
}

int readIntRegister(int deviceHandle, int Register, int* Value) {
	int r;
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[READ_REGISTER_SEND_LENGTH], recBuffer[READ_DOUBLE_REGISTER_REC_LENGTH], iBuffer[4];
    
    int* iPointer; // pointer for casting bytes to an int.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 57;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 6;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Read Holding Registers ( function # 3 )
	sendBuffer[9] = 3;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	//Bytes 10 and 11 are NumRegs, in int's case: 2
	sendBuffer[12] = 0;
	sendBuffer[13] = 2;
	
	r = LJUSB_Write( devHandle, sendBuffer, READ_REGISTER_SEND_LENGTH );
    
    if( r != READ_REGISTER_SEND_LENGTH ) {
        return -1; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, READ_DOUBLE_REGISTER_REC_LENGTH );
	
    if( r != READ_DOUBLE_REGISTER_REC_LENGTH ) {
        return -2; //Error in read
    }
    
    iBuffer[0] = recBuffer[12];
    iBuffer[1] = recBuffer[11];
    iBuffer[2] = recBuffer[10];
    iBuffer[3] = recBuffer[9];
    
    iPointer = (int*)iBuffer;
	
    *Value = *iPointer;
	
	return 0;
}

int readShortRegister(int deviceHandle, int Register, short* Value){
	int r;
	
    HANDLE devHandle = (HANDLE)deviceHandle;
	
    BYTE sendBuffer[READ_REGISTER_SEND_LENGTH], recBuffer[READ_SINGLE_REGISTER_REC_LENGTH], sBuffer[2];
    
    short* sPointer; // pointer for casting bytes to short.
	
	// Have to add 0, 0 to front.
	sendBuffer[0] = 0;
	sendBuffer[1] = 0;
	
	// Bytes 0 and 1 are TransID
	sendBuffer[2] = 0;
	sendBuffer[3] = 59;
	
	// Bytes 2 and 3 are ProtocolID, set to 0
	sendBuffer[4] = 0;
	sendBuffer[5] = 0;
	
	// Bytes 4 and 5 are length.
	sendBuffer[6] = 0;
	sendBuffer[7] = 6;
	
	// Byte 6 is Unit ID.
	sendBuffer[8] = 0;
	
	// Byte 7 is Read Holding Registers ( function # 3 )
	sendBuffer[9] = 3;
	
	// Bytes 8 and 9 are Address
	sendBuffer[10] = (BYTE)(((Register) >> 8) & 0xff);
	sendBuffer[11] = (BYTE)((Register) & 0xff);
	
	//Bytes 10 and 11 are NumRegs, for shorts always 1
	sendBuffer[12] = 0;
	sendBuffer[13] = 1;
	
	r = LJUSB_Write( devHandle, sendBuffer, READ_REGISTER_SEND_LENGTH );
    
    if( r != READ_REGISTER_SEND_LENGTH ) {
        return -1; //Error in write
    }
	
	
	r = LJUSB_Read( devHandle, recBuffer, READ_SINGLE_REGISTER_REC_LENGTH );
	
    if( r != READ_SINGLE_REGISTER_REC_LENGTH ) {
        return -2; //Error in read
    }
    
    sBuffer[0] = recBuffer[10];
    sBuffer[1] = recBuffer[9];
    
    sPointer = (short*)sBuffer;
	
    *Value = (short)*sPointer;
	
	return 0;
}

